import greenfoot.*;  // (World, Actor, GreenfootImage, Greenfoot and MouseInfo)

/**
 * 
 * 
 * @author Joseph Lenton
 */
public class Player  extends HealthActor
{
    private static final GamePad.Button BUTTON_USE_SPECIAL = GamePad.Button.L1;
    private static final GamePad.Button SHOOT_BUTTON = GamePad.Button.R1;
    
    private static final int WEAPON_X_OFFSET = 20;
    private static final int WEAPON_Y_OFFSET = 7;
    
    private static final int START_HP = 20;
    
    private static final float SPEED = 3.8f;
    
    // The minimum amount of movement across the combination of x and y axis
    // for movement to exist.
    private static final float MINIMUM_MOVEMENT = 0.1f;
    
    private final GamePad pad;
    private Weapon weapon;
    private Weapon specialWeapon;
    private boolean shootDown;
    
    public Player(GamePad pad)
    {
        super( START_HP );
        
        this.pad = pad;
        weapon = new WeaponShotgun();
        specialWeapon = new WeaponSpecialShot();
        shootDown = false;
    }
    
    /**
     * 
     */
    public void addedToWorld(World world)
    {
        super.addedToWorld(world);
        
        world.addObject( new OverlayMain(), 0, 0 );
    }
    
    @Override
    public void move( float moveX, float moveY )
    {
        super.move( moveX, moveY );
        
        if ( getOneIntersectingObject(Enemy.class) != null ) {
            super.move( -moveX, -moveY );
        }
    }
    
    @Override
    public void damage(int damage)
    {
        if ( !isDamaged() ) {
            super.damage( damage );
        }
    }
    
    /**
     * 
     */
    public void act() 
    {
        super.act();
        
        final Direction moveDir = getMove();
        moveAngle( moveDir.getAngle(), SPEED*moveDir.getStrength() );
        
        final Direction shootAngle = getShootAngle();
        setRotation( shootAngle.getAngle() );
        
        final boolean useSpecial = isUsingSpecial();
        if ( useSpecial && specialWeapon.shoot(this) ) {
            // do nothing, the work is performed in the condition
        } else if ( isShooting() ) {
            if ( weapon.isRapidFire() || !shootDown ) {
                weapon.shoot( this );
                shootDown = true;
            }
        } else {
            shootDown = false;
        }
    }
    
    @Override
    public void remove()
    {
        final Game game = (Game) getWorld();
        game.endGame();
        
        super.remove();
    }
    
    /**
     * The Player might not have a game pad during development to avoid the
     * need to restart Greenfoot repeatedly.
     * @return True or false to state if this has a gamepad or not.
     */
    private boolean hasGamePad()
    {
        return pad != null;
    }
    
    private boolean isUsingSpecial()
    {
        return Greenfoot.isKeyDown("z") ||
                ( hasGamePad() && pad.isDown(BUTTON_USE_SPECIAL) );
    }
    
    private boolean isMovement(float moveX, float moveY)
    {
        return ( Math.abs(moveX) + Math.abs(moveY) ) > MINIMUM_MOVEMENT;
    }
    
    /**
     * @return Null if this is not shooting, otherwise the Direction to shoot in.
     */
    private Direction getShootAngle()
    {
        // check keyboard
        if ( Greenfoot.isKeyDown("right") ) {
            return new Direction(0, 1f);
        } else if ( Greenfoot.isKeyDown("down") ) {
            return new Direction(90, 1f);
        } else if ( Greenfoot.isKeyDown("left") ) {
            return new Direction(180, 1f);
        } else if ( Greenfoot.isKeyDown("up") ) {
            return new Direction(270, 1f);
        }
        
        // check gamepad
        if ( hasGamePad() ) {
            return pad.getAxis( GamePad.Axis.RIGHT );
        }
        
        // no shooting
        return null;
    }
    
    private boolean isShooting()
    {
        return Greenfoot.isKeyDown("space") || (hasGamePad() && pad.isDown( SHOOT_BUTTON ));
    }
    
    private Direction getMove()
    {
        final float keyMoveX = getKeyHorizontalMove();
        final float keyMoveY = getKeyVerticalMove();
        
        // keyboard movement
        if ( keyMoveX != 0.0f || keyMoveY != 0.0f ) {
            return new Direction( (int) Math.toDegrees(Math.atan2(keyMoveY, keyMoveX)), 1f );
        // joypad movement
        } else if ( hasGamePad() ) {
            return pad.getAxis( GamePad.Axis.LEFT );
        // neither
        } else {
            return null;
        }
    }
    
    private float getKeyVerticalMove()
    {
        if ( Greenfoot.isKeyDown("s") ) {
            return 1f;
        } else if ( Greenfoot.isKeyDown("w") ) {
            return -1f;
        } else {
            return 0f;
        }
    }
    
    private float getKeyHorizontalMove()
    {
        if ( Greenfoot.isKeyDown("a") ) {
            return -1f;
        } else if ( Greenfoot.isKeyDown("d") ) {
            return 1f;
        } else {
            return 0f;
        }
    }
    
    /**
     * @return the X co-ordinate of where bullets should start from.
     */
    public int getWeaponX()
    {
        final double angle = Math.toRadians( getRotation() );
        return getX() + (int) ( WEAPON_X_OFFSET*Math.cos(angle) - WEAPON_Y_OFFSET*Math.sin(angle) );
    }
    
    public int getWeaponY()
    {
        final double angle = Math.toRadians( getRotation() );
        return getY() + (int) ( WEAPON_Y_OFFSET*Math.sin(angle) + WEAPON_Y_OFFSET*Math.cos(angle) );
    }
}
