import greenfoot.*;  // (World, Actor, GreenfootImage, Greenfoot and MouseInfo)
import java.awt.Color;
import java.util.Map;
import java.util.HashMap;

/**
 * An actor with added health properties.
 * 
 * @author Joseph Lenton
 */
public abstract class HealthActor  extends GameActor
{
    private static final Map<Class<?>, GreenfootImage> DAMAGE_IMAGES = new HashMap<Class<?>, GreenfootImage>();
    private static final Map<Class<?>, GreenfootImage> NORMAL_IMAGES = new HashMap<Class<?>, GreenfootImage>();
    
    private static GreenfootImage getDamageImage(Actor actor)
    {
        final Class<?> klass = actor.getClass();
        GreenfootImage damageImage = DAMAGE_IMAGES.get( klass );
        
        if ( damageImage == null ) {
            final GreenfootImage img = actor.getImage();
            damageImage = generateDamageImage( img );
            DAMAGE_IMAGES.put( klass, damageImage );
            NORMAL_IMAGES.put( klass, img );
        }
        
        return damageImage;
    }
    
    private static GreenfootImage generateDamageImage(GreenfootImage img)
    {
        final GreenfootImage damageImage = new GreenfootImage( img );
        
        for ( int y = 0; y < damageImage.getHeight(); y++ ) {
            for ( int x = 0; x < damageImage.getWidth(); x++ ) {
                final Color pixel = damageImage.getColorAt( x, y );
                damageImage.setColorAt( x, y, new Color(255, 255, 255, pixel.getAlpha()) );
            }
        }
        
        return damageImage;
    }
    
    private static final long DAMAGE_DELAY = 50;
    
    private final GreenfootImage damageImage;
    private boolean isDamaged;
    private boolean pastMidDamage;
    private long damageTime;
    private int maxHP;
    private int hp;
    
    /**
     * Creates a new actor with the starting hp.
     * @param hp The starting hp of this actor.
     */
    public HealthActor(int hp)
    {
        this.damageImage = HealthActor.getDamageImage( this );
        setMaxHP( hp );
    }
    
    /**
     * Replaces the maximum HP of this actor to the amount given.
     * It also fully heals the actor, as their current HP is set
     * to the maximum too.
     * 
     * @param maxHP The new maximum hp of this actor, must be 0 or greater.
     */
    protected final void setMaxHP(int maxHP)
    {
        if ( maxHP <= 0 ) {
            throw new IllegalArgumentException("The starting hp must be greater then 0, was given: " + maxHP);
        }
        
        this.maxHP = maxHP;
        this.hp = maxHP;
    }
    
    /**
     * Decreases the health of the damage by the given amount.
     * Negative damage will heal the actor.
     * If the hp reaches 0 then it will call to remove itself.
     * 
     * @param damage The amount to remove from the actors hp.
     */
    public void damage(int damage)
    {
        hp -= damage;
        
        if ( hp > maxHP ) {
            hp = maxHP;
        } else if ( hp < 0 ) {
            hp = 0;
            remove();
        }
        
        isDamaged = true;
        damageTime = System.currentTimeMillis() + DAMAGE_DELAY;
        setImage( damageImage );
    }
    
    public boolean isDamaged()
    {
        return isDamaged;
    }
    
    /**
     * @return The starting maximum hp of the actor.
     */
    public int getMaxHP()
    {
        return maxHP;
    }
    
    /**
     * @return The current hp of the actor.
     */
    public int getHP()
    {
        return hp;
    }
    
    protected void actDamageImage()
    {
        if ( isDamaged && damageTime < System.currentTimeMillis() ) {
            if ( !pastMidDamage ) {
                pastMidDamage = true;
                setImage( NORMAL_IMAGES.get(getClass()) );
                damageTime = System.currentTimeMillis() + DAMAGE_DELAY;
            } else {
                isDamaged = false;
                pastMidDamage = false;
            }
        }
    }
    
    public void act()
    {
        actDamageImage();
        super.act();
    }
}
