import greenfoot.*;  // (World, Actor, GreenfootImage, Greenfoot and MouseInfo)

/**
 * Defines a list of helper functions for game objects.
 * 
 * @author Joseph Lenton
 */
public abstract class GameActor  extends Actor
{
    private float x;
    private float y;
    
    private boolean isMoving;
    private float moveX, moveY;
    
    @Override
    public void addedToWorld(World world)
    {
        this.x = getX();
        this.y = getY();
    }
    
    @Override
    public Game getWorld()
    {
        return (Game) super.getWorld();
    }
    
    public float getXf()
    {
        return x;
    }
    
    public float getYf()
    {
        return y;
    }
    
    public void setLocation(float x, float y)
    {
        setLocationF( x, y );
        super.setLocation( Math.round(x), Math.round(y) );
    }
    
    public void setLocation(int x, int y)
    {
        setLocationF( x, y );
        super.setLocation( x, y );
    }
    
    private void setLocationF(float x, float y)
    {
        this.x = Math.max( 0, Math.min(x, Game.WIDTH-1) );
        this.y = Math.max( 0, Math.min(y, Game.HEIGHT-1) );
    }
    
    /**
     * @param angle The angle to move in, in degrees.
     * @param move The amount to move in the given angle.
     */
    public void moveAngle(float angle, float move)
    {
        final double dblAngle = Math.toRadians( angle );
        move(
                move * (float)Math.cos( dblAngle ),
                move * (float)Math.sin( dblAngle )
        );
    }
    
    /**
     * Moves this actor across in the x and y directions by the given amount.
     * @param moveX The amount to move in the x direction.
     * @param moveY The amount to move in the y direction.
     */
    public void move(float moveX, float moveY)
    {
        final float delta = getWorld().getDeltaTime();
        setLocation( getXf() + delta*moveX, getYf() + delta*moveY );
    }
    
    /**
     * Removes this object from the world it is within.
     */
    public void remove()
    {
        getWorld().removeObject( this );
    }
    
    public float getMoveX()
    {
        return moveX;
    }
    
    public float getMoveY()
    {
        return moveY;
    }
    
    public boolean isMoving()
    {
        return isMoving;
    }
    
    public void stopMoving()
    {
        isMoving = false;
    }
    
    public void setMoving(float moveX, float moveY)
    {
        this.moveX = moveX;
        this.moveY = moveY;
        
        isMoving = true;
    }
    
    public void setMoveAngle(double angle, float speed)
    {
        setMoving(
                (float) ( speed*Math.cos(angle) ),
                (float) ( speed*Math.sin(angle) )
        );
        setRotation( (int) Math.toDegrees(angle) );
    }
    
    public boolean isOnScreenEdge()
    {
        final int x = getX();
        final int y = getY();
        
        return x == 0 || x == Game.WIDTH-1 ||
               y == 0 || y == Game.HEIGHT-1 ;
    }
    
    @Override
    public void act()
    {
        super.act();
        actMove();
    }
    
    protected void actMove()
    {
        if ( isMoving() ) {
            move( getMoveX(), getMoveY() );
        }
    }
    
    protected double getAngle(Actor to)
    {
        return getAngle( this, to );
    }
    
    protected double getAngle(Actor from, Actor to)
    {
        final int diffX = to.getX()-from.getX();
        final int diffY = to.getY()-from.getY();
        
        return Math.atan2( diffY, diffX );
    }
}
