import greenfoot.*;  // (World, Actor, GreenfootImage, Greenfoot and MouseInfo)
import java.util.List;

/**
 * Write a description of class Game here.
 * 
 * @author Joseph Lenton
 */
public class Game  extends World
{
    public static final int WIDTH = 800;
    public static final int HEIGHT = 480;
    
    public static final int START_X = WIDTH/2;
    public static final int START_Y = HEIGHT/2;
    
    /**
     * States if this Game will be using the controller or not.
     * 
     * Typically this is set to false during development and true
     * for testing and production.
     */
    private static final boolean USE_CONTROLLER = true;
    
    private static final int PRESS_START_X = WIDTH/2;
    private static final int PRESS_START_Y = 415;
    
    private static final int TITLE_START_X = WIDTH/2;
    private static final int TITLE_START_Y = HEIGHT/2 - 50;
    
    private static final long DELAY_TIME = 6000;
    private static final long MIN_DELAY_TIME = 2000;
    private static final double DELAY_MULTIPLYER = 0.99;
    private static final float HEALTH_INCREMENT = 0.6f;
    
    private final DeltaTime delta;
    private final SpawnInfo[] spawns;
    private final GamePad playerPad;
    
    private float healthAdd;
    private boolean isPlaying;
    private long delayTime;
    private long delay;
    private int round;
    private Score score;
    
    /**
     * Constructor for objects of class Game.
     */
    public Game()
    {
        super(WIDTH, HEIGHT, 1);
        
        this.delta = new DeltaTime();
        this.playerPad = USE_CONTROLLER ? GamePad.getGamePad() : null ;
        this.spawns = new SpawnInfo[] {
            new SpawnInfo( 3, Tomb.class ),
            new SpawnInfo( 1, Grave.class )
        };
        
        setActOrder(
                Enemy.class,
                Player.class,
                Overlay.class
        );
        
        setPaintOrder(
                PressStart.class,
                Title.class,
                Score.class,
                Overlay.class,
                Player.class,
                Spawner.class,
                Enemy.class,
                PlayerBullet.class
        );
        
        startTitle();
    }
    
    public float getDeltaTime()
    {
        return delta.getDeltaTime();
    }
    
    /**
     * Removes all of the actors contained within this world, from the world.
     */
    private final void removeAll()
    {
        removeObjects( getObjects(null) );
    }
    
    private final void startTitle()
    {
        removeAll();
        addObject( new Title(), TITLE_START_X, TITLE_START_Y );
        showPressStart();
    }
    
    public final void startGame()
    {
        removeAll();
        final Player player = new Player( playerPad );
        
        addObject( player, START_X, START_Y );
        delayTime = DELAY_TIME;
        delay = System.currentTimeMillis() + delayTime;
        round = 1;
        healthAdd = 1;
        
        delta.reset();
        score = new Score();
        addObject( score, 0, 0 ); // score sets its own location
        isPlaying = true;
    }
    
    public int getHealthAdd()
    {
        return Math.round( healthAdd );
    }
    
    public final void endGame()
    {
        showPressStart();
        isPlaying = false;
    }
    
    private final void showPressStart()
    {
        addObject( new PressStart(playerPad), PRESS_START_X, PRESS_START_Y );
    }
    
    /* ### Runtime Act and update code. ### */
    
    /**
     * 
     */
    @Override
    public void act()
    {
        if ( isPlaying ) {
            actGame();
        }
        
        delta.update();
    }
    
    /**
     * Updates the main game itself.
     */
    protected void actGame()
    {
        final long now = System.currentTimeMillis();
        
        if ( delay < now ) {
            for ( SpawnInfo sInfo : spawns ) {
                if ( sInfo.isSpawn(round) ) {
                    // make the spawner
                    final Actor newSpawner = sInfo.spawn();
                    int sx = 0, sy = 0;
                    
                    /* This section of code is to ensure the spawner is not appearing
                     * on top of the Player. */
                    final List ls = getObjects( Player.class );
                    final Player player;
                    
                    if ( ls.size() == 0 ) {
                        return;
                    } else {
                        player = (Player) ls.get(0);
                    }
                    
                    final int px = player.getX();
                    final int py = player.getY();
                    final GreenfootImage pImg = player.getImage();
                    final GreenfootImage sImg = newSpawner.getImage();
                    final int width = getWidth();
                    final int height = getHeight();
                    
                    do {
                        sx = Greenfoot.getRandomNumber( width );
                        sy = Greenfoot.getRandomNumber( height );
                    } while ( !isOverlap(px, py, pImg, sx, sy, sImg) );
                    
                    // then we add, and are done
                    addObject( newSpawner, sx, sy );
                    break;
                }
            }
            
            // update round delay info, regardless of if we spawn or not
            delayTime = Math.max(
                    MIN_DELAY_TIME,
                    (long) (delayTime*DELAY_MULTIPLYER)
            );
            healthAdd += HEALTH_INCREMENT;
            delay = now + delayTime;
            round++;
        }
    }
    
    private static boolean isOverlap(int x1, int y1, GreenfootImage img1, int x2, int y2, GreenfootImage img2 )
    {
        return (x1-x2) < (img1.getWidth()/2 + img2.getWidth()/2) &&
               (y1-y2) < (img1.getHeight()/2 + img2.getHeight()/2);
    }
    
    /**
     * If there is a Score in the Game, then it is increment by the given amount.
     * Otherwise nothing will happen.
     * 
     * @param amount The amount to increment the current score by.
     */
    public void incrementScore(int amount)
    {
        if ( score != null ) {
            score.increment( amount );
        }
    }
    
    /**
     * @return True if this world is currently updating it's game contents, otherwise false.
     */
    public boolean isPlaying()
    {
        return isPlaying;
    }
    
    /**
     * Simple class for storing spawn information together.
     * This info contains the multiple of the round for enemies
     * to be shown on, and the class to use for spawning that enemy.
     * 
     * The class must contain a no-args constructor.
     */
    private static class SpawnInfo
    {
        private final int mod;
        private final Class<? extends Actor> spawnClass;
        
        public SpawnInfo(int mod, Class<? extends Actor> spawnClass)
        {
            this.mod = mod;
            this.spawnClass = spawnClass;
        }
        
        public boolean isSpawn(int round)
        {
            return (round % mod) == 0;
        }
        
        public Actor spawn()
        {
            try {
                return spawnClass.newInstance();
            } catch (IllegalAccessException ex) {
                throw new RuntimeException( ex );
            } catch (InstantiationException ex) {
                throw new RuntimeException( ex );
            }
        }
    }
}
