/*
 * Decompiled with CFR 0.152.
 */
package org.junit.platform.reporting.legacy.xml;

import java.io.Writer;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.MessageFormat;
import java.text.NumberFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.TreeSet;
import java.util.stream.Collectors;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import org.junit.platform.commons.util.ExceptionUtils;
import org.junit.platform.commons.util.StringUtils;
import org.junit.platform.engine.TestExecutionResult;
import org.junit.platform.engine.reporting.ReportEntry;
import org.junit.platform.launcher.TestIdentifier;
import org.junit.platform.launcher.TestPlan;
import org.junit.platform.launcher.listeners.LegacyReportingUtils;
import org.junit.platform.reporting.legacy.xml.XmlReportData;

class XmlReportWriter {
    private static final String CDATA_START = "<![CDATA[";
    private static final String CDATA_END = "]]>";
    private final XmlReportData reportData;

    XmlReportWriter(XmlReportData reportData) {
        this.reportData = reportData;
    }

    void writeXmlReport(TestIdentifier testIdentifier, Writer out) throws XMLStreamException {
        List<TestIdentifier> tests = this.reportData.getTestPlan().getDescendants(testIdentifier).stream().filter(TestIdentifier::isTest).collect(Collectors.toList());
        this.writeXmlReport(testIdentifier, tests, out);
    }

    private void writeXmlReport(TestIdentifier testIdentifier, List<TestIdentifier> tests, Writer out) throws XMLStreamException {
        XMLOutputFactory factory = XMLOutputFactory.newInstance();
        XMLStreamWriter xmlWriter = factory.createXMLStreamWriter(out);
        xmlWriter.writeStartDocument("UTF-8", "1.0");
        this.newLine(xmlWriter);
        this.writeTestsuite(testIdentifier, tests, xmlWriter);
        xmlWriter.writeEndDocument();
        xmlWriter.flush();
        xmlWriter.close();
    }

    private void writeTestsuite(TestIdentifier testIdentifier, List<TestIdentifier> tests, XMLStreamWriter writer) throws XMLStreamException {
        NumberFormat numberFormat = NumberFormat.getInstance(Locale.US);
        writer.writeStartElement("testsuite");
        this.writeSuiteAttributes(testIdentifier, tests, numberFormat, writer);
        this.newLine(writer);
        this.writeSystemProperties(writer);
        for (TestIdentifier test : tests) {
            this.writeTestcase(test, numberFormat, writer);
        }
        this.writeOutputElement("system-out", this.formatNonStandardAttributesAsString(testIdentifier), writer);
        writer.writeEndElement();
        this.newLine(writer);
    }

    private void writeSuiteAttributes(TestIdentifier testIdentifier, List<TestIdentifier> tests, NumberFormat numberFormat, XMLStreamWriter writer) throws XMLStreamException {
        writer.writeAttribute("name", testIdentifier.getDisplayName());
        this.writeTestCounts(tests, writer);
        writer.writeAttribute("time", this.getTime(testIdentifier, numberFormat));
        writer.writeAttribute("hostname", this.getHostname().orElse("<unknown host>"));
        writer.writeAttribute("timestamp", DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(this.getCurrentDateTime()));
    }

    private void writeTestCounts(List<TestIdentifier> tests, XMLStreamWriter writer) throws XMLStreamException {
        TestCounts testCounts = TestCounts.from(this.reportData, tests);
        writer.writeAttribute("tests", String.valueOf(testCounts.getTotal()));
        writer.writeAttribute("skipped", String.valueOf(testCounts.getSkipped()));
        writer.writeAttribute("failures", String.valueOf(testCounts.getFailures()));
        writer.writeAttribute("errors", String.valueOf(testCounts.getErrors()));
    }

    private void writeSystemProperties(XMLStreamWriter writer) throws XMLStreamException {
        writer.writeStartElement("properties");
        this.newLine(writer);
        Properties systemProperties = System.getProperties();
        for (String propertyName : new TreeSet<String>(systemProperties.stringPropertyNames())) {
            writer.writeEmptyElement("property");
            writer.writeAttribute("name", propertyName);
            writer.writeAttribute("value", systemProperties.getProperty(propertyName));
            this.newLine(writer);
        }
        writer.writeEndElement();
        this.newLine(writer);
    }

    private void writeTestcase(TestIdentifier testIdentifier, NumberFormat numberFormat, XMLStreamWriter writer) throws XMLStreamException {
        writer.writeStartElement("testcase");
        writer.writeAttribute("name", this.getName(testIdentifier));
        writer.writeAttribute("classname", this.getClassName(testIdentifier));
        writer.writeAttribute("time", this.getTime(testIdentifier, numberFormat));
        this.newLine(writer);
        this.writeSkippedOrErrorOrFailureElement(testIdentifier, writer);
        ArrayList<String> systemOutElements = new ArrayList<String>();
        ArrayList<String> systemErrElements = new ArrayList<String>();
        systemOutElements.add(this.formatNonStandardAttributesAsString(testIdentifier));
        this.collectReportEntries(testIdentifier, systemOutElements, systemErrElements);
        this.writeOutputElements("system-out", systemOutElements, writer);
        this.writeOutputElements("system-err", systemErrElements, writer);
        writer.writeEndElement();
        this.newLine(writer);
    }

    private String getName(TestIdentifier testIdentifier) {
        return testIdentifier.getLegacyReportingName();
    }

    private String getClassName(TestIdentifier testIdentifier) {
        return LegacyReportingUtils.getClassName((TestPlan)this.reportData.getTestPlan(), (TestIdentifier)testIdentifier);
    }

    private void writeSkippedOrErrorOrFailureElement(TestIdentifier testIdentifier, XMLStreamWriter writer) throws XMLStreamException {
        if (this.reportData.wasSkipped(testIdentifier)) {
            this.writeSkippedElement(this.reportData.getSkipReason(testIdentifier), writer);
        } else {
            Optional<TestExecutionResult> result = this.reportData.getResult(testIdentifier);
            if (result.isPresent() && result.get().getStatus() == TestExecutionResult.Status.FAILED) {
                this.writeErrorOrFailureElement(result.get(), writer);
            }
        }
    }

    private void writeSkippedElement(String reason, XMLStreamWriter writer) throws XMLStreamException {
        if (StringUtils.isNotBlank((String)reason)) {
            writer.writeStartElement("skipped");
            this.writeCDataSafely(writer, reason);
            writer.writeEndElement();
        } else {
            writer.writeEmptyElement("skipped");
        }
        this.newLine(writer);
    }

    private void writeErrorOrFailureElement(TestExecutionResult result, XMLStreamWriter writer) throws XMLStreamException {
        Optional throwable = result.getThrowable();
        if (throwable.isPresent()) {
            writer.writeStartElement(XmlReportWriter.isFailure(result) ? "failure" : "error");
            this.writeFailureAttributesAndContent((Throwable)throwable.get(), writer);
            writer.writeEndElement();
        } else {
            writer.writeEmptyElement("error");
        }
        this.newLine(writer);
    }

    private void writeFailureAttributesAndContent(Throwable throwable, XMLStreamWriter writer) throws XMLStreamException {
        if (throwable.getMessage() != null) {
            writer.writeAttribute("message", throwable.getMessage());
        }
        writer.writeAttribute("type", throwable.getClass().getName());
        this.writeCDataSafely(writer, ExceptionUtils.readStackTrace((Throwable)throwable));
    }

    private void collectReportEntries(TestIdentifier testIdentifier, List<String> systemOutElements, List<String> systemErrElements) {
        List<ReportEntry> entries = this.reportData.getReportEntries(testIdentifier);
        if (!entries.isEmpty()) {
            ArrayList<String> systemOutElementsForCapturedOutput = new ArrayList<String>();
            StringBuilder formattedReportEntries = new StringBuilder();
            for (int i = 0; i < entries.size(); ++i) {
                ReportEntry reportEntry = entries.get(i);
                LinkedHashMap<String, String> keyValuePairs = new LinkedHashMap<String, String>(reportEntry.getKeyValuePairs());
                this.removeIfPresentAndAddAsSeparateElement(keyValuePairs, "stdout", systemOutElementsForCapturedOutput);
                this.removeIfPresentAndAddAsSeparateElement(keyValuePairs, "stderr", systemErrElements);
                if (keyValuePairs.isEmpty()) continue;
                this.buildReportEntryDescription(reportEntry.getTimestamp(), keyValuePairs, i + 1, formattedReportEntries);
            }
            systemOutElements.add(formattedReportEntries.toString().trim());
            systemOutElements.addAll(systemOutElementsForCapturedOutput);
        }
    }

    private void removeIfPresentAndAddAsSeparateElement(Map<String, String> keyValuePairs, String key, List<String> elements) {
        String value = keyValuePairs.remove(key);
        if (value != null) {
            elements.add(value);
        }
    }

    private void buildReportEntryDescription(LocalDateTime timestamp, Map<String, String> keyValuePairs, int entryNumber, StringBuilder result) {
        result.append(MessageFormat.format("Report Entry #{0} (timestamp: {1})\n", entryNumber, DateTimeFormatter.ISO_LOCAL_DATE_TIME.format(timestamp)));
        keyValuePairs.forEach((key, value) -> result.append(MessageFormat.format("\t- {0}: {1}\n", key, value)));
    }

    private String getTime(TestIdentifier testIdentifier, NumberFormat numberFormat) {
        return numberFormat.format(this.reportData.getDurationInSeconds(testIdentifier));
    }

    private Optional<String> getHostname() {
        try {
            return Optional.ofNullable(InetAddress.getLocalHost().getHostName());
        }
        catch (UnknownHostException e) {
            return Optional.empty();
        }
    }

    private LocalDateTime getCurrentDateTime() {
        return LocalDateTime.now(this.reportData.getClock()).withNano(0);
    }

    private String formatNonStandardAttributesAsString(TestIdentifier testIdentifier) {
        return "unique-id: " + testIdentifier.getUniqueId() + "\ndisplay-name: " + testIdentifier.getDisplayName();
    }

    private void writeOutputElements(String elementName, List<String> elements, XMLStreamWriter writer) throws XMLStreamException {
        for (String content : elements) {
            this.writeOutputElement(elementName, content, writer);
        }
    }

    private void writeOutputElement(String elementName, String content, XMLStreamWriter writer) throws XMLStreamException {
        writer.writeStartElement(elementName);
        this.writeCDataSafely(writer, "\n" + content + "\n");
        writer.writeEndElement();
        this.newLine(writer);
    }

    private void writeCDataSafely(XMLStreamWriter writer, String data) throws XMLStreamException {
        writer.writeCData(data.replace(CDATA_END, "]]]]><![CDATA[>"));
    }

    private void newLine(XMLStreamWriter xmlWriter) throws XMLStreamException {
        xmlWriter.writeCharacters("\n");
    }

    private static boolean isFailure(TestExecutionResult result) {
        Optional throwable = result.getThrowable();
        return throwable.isPresent() && throwable.get() instanceof AssertionError;
    }

    private static class TestCounts {
        private final long total;
        private long skipped;
        private long failures;
        private long errors;

        static TestCounts from(XmlReportData reportData, List<TestIdentifier> tests) {
            TestCounts counts = new TestCounts(tests.size());
            for (TestIdentifier test : tests) {
                if (reportData.wasSkipped(test)) {
                    ++counts.skipped;
                    continue;
                }
                Optional<TestExecutionResult> result = reportData.getResult(test);
                if (!result.isPresent() || result.get().getStatus() != TestExecutionResult.Status.FAILED) continue;
                if (XmlReportWriter.isFailure(result.get())) {
                    ++counts.failures;
                    continue;
                }
                ++counts.errors;
            }
            return counts;
        }

        TestCounts(long total) {
            this.total = total;
        }

        public long getTotal() {
            return this.total;
        }

        public long getSkipped() {
            return this.skipped;
        }

        public long getFailures() {
            return this.failures;
        }

        public long getErrors() {
            return this.errors;
        }
    }
}

